# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module Report
      module GroupIssues
        class FailureProcessor
          DEFAULT_MIN_FAILURES = 2

          def initialize(options = {})
            @options = options
            @pattern_matcher = options[:pattern_matcher] || ErrorPatternMatcher.new
            @normalizer = options[:normalizer] || ErrorMessageNormalizer.new
            @config = options[:config] || {}
          end

          def process_failures(failures, &)
            Runtime::Logger.info "Processing #{failures.size} failures for grouping..."
            grouped_failures = {}

            failures.each do |failure|
              process_single_failure(failure, grouped_failures)
            end

            Runtime::Logger.info "Found #{grouped_failures.size} groups before filtering"
            grouped_failures.each_value(&)
          end

          def filter_groups_by_threshold(grouped_failures)
            min_failures = @config.dig(:thresholds, :min_failures_to_group) || DEFAULT_MIN_FAILURES

            grouped_failures.select! do |_fingerprint, grouped_failure|
              grouped_failure[:failures].size >= min_failures
            end

            Runtime::Logger.info "Found #{grouped_failures.size} groups after filtering"
          end

          private

          def process_single_failure(failure, grouped_failures)
            error_message = failure.dig(:exception, 'message') || failure.dig(:exceptions, 0, 'message')
            Runtime::Logger.info "Processing failure: #{failure[:description]}"
            Runtime::Logger.info "Error message: #{error_message[0..100]}..." if error_message

            return unless error_message && @pattern_matcher.environment_error?(error_message)

            Runtime::Logger.info "Identified as environment error"
            group_environment_failure(failure, error_message, grouped_failures)
          end

          def group_environment_failure(failure, error_message, grouped_failures)
            normalized_message = @normalizer.normalize(error_message)
            fingerprint = @normalizer.create_fingerprint(normalized_message)
            pattern_name = @pattern_matcher.pattern_name(error_message)

            grouped_failures[fingerprint] ||= build_grouped_failure(fingerprint, pattern_name, normalized_message)
            grouped_failures[fingerprint][:failures] << failure
          end

          def build_grouped_failure(fingerprint, pattern_name, normalized_message)
            {
              fingerprint: fingerprint,
              pattern_name: pattern_name,
              normalized_message: normalized_message,
              failures: []
            }
          end
        end
      end
    end
  end
end
